function plot(ssm, de)
%PLOT Plot a StateSpaceModel or a DynamicElement belonging to a StateSpaceModel
%
% Usage:
%   plot(ssm)
%   plot(ssm, de)
%
% If only ssm is given, creates 4 different figures:
%   Figure 1: plots all state trajectories
%   Figure 2: plots all control trajectories
%   Figure 3: plots all input trajectories
%   Figure 4: plots all auxiliary state trajectories
%
% If de is also given, creates a plot of the DynamicElement de.
% In this case this function is useful if de has no val field calculated,
% but a def field that depends on other elements in ssm, that do have a val
% field calculated

% David Katzin, Wageningen University
% david.katzin@wur.nl

    if ~exist('de', 'var') % plot the whole StateSpaceModel
        inputNames = fieldnames(ssm.d);
        stateNames = fieldnames(ssm.x);
        auxNames = fieldnames(ssm.a);
        if isempty(ssm.u)
            ctrlNames = [];
        else
            ctrlNames = fieldnames(ssm.u);
        end

        inputNum = length(inputNames);
        stateNum = length(stateNames);
        ctrlNum = length(ctrlNames);
        auxNum = length(auxNames);

        figure;
        for n=1:stateNum
            subplot(stateNum,1,n);
            plot(ssm.x.(stateNames{n}));
            title(['x.' stateNames{n}]);
            axis tight;
        end

        if ctrlNum>0
            figure;
            for n=1:ctrlNum
                subplot(ctrlNum,1,n);
                plot(ssm.u.(ctrlNames{n}));
                title(['u.' ctrlNames{n}]);
                axis tight;
            end
        end

        figure;
        for n=1:inputNum
            subplot(inputNum,1,n);
            plot(ssm.d.(inputNames{n}));
            title(['d.' inputNames{n}]);
            axis tight;
        end

        figure;
        for n=1:auxNum
            subplot(auxNum,1,n);
            plot(ssm.a.(auxNames{n}));
            title(['a.' auxNames{n}]);
            axis tight;
        end
        
    else % plot a dynamic element belonging to this ssm
        stateNames = fieldnames(ssm.x);
        for k=1:length(stateNames)
            x.(stateNames{k}) = ssm.x.(stateNames{k});
            x.(stateNames{k}).def = ['x.' (stateNames{k})];
        end
        u = ssm.u; d = ssm.d; a = ssm.a; p = ssm.p;
        plot(eval(de.def));
    end
end

